// Copyright RedShift PASYS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "EditorSubsystem.h"
#include "RedShiftPASYSSubsystem.generated.h"

/**
 * Editor subsystem for RedShift Performance Analysis System (PASYS).
 * Logs and helps optimize game, textures, shaders; identifies solvable bottlenecks.
 */
UCLASS()
class REDSHIFTPASYS_API URedShiftPASYSSubsystem : public UEditorSubsystem
{
	GENERATED_BODY()

public:
	/** Run full optimization report: textures, materials/shaders, and bottleneck summary. Logs to Output Log and optional file. */
	UFUNCTION(BlueprintCallable, Category = "RedShift PASYS", meta = (DisplayName = "Run Full Optimization Report"))
	void RunFullOptimizationReport();

	/** Audit textures: list large/non-streaming textures and suggest fixes. */
	UFUNCTION(BlueprintCallable, Category = "RedShift PASYS", meta = (DisplayName = "Run Texture Audit"))
	void RunTextureAudit();

	/** Audit materials/shaders: list complex materials and suggest simplifications. */
	UFUNCTION(BlueprintCallable, Category = "RedShift PASYS", meta = (DisplayName = "Run Shader/Material Audit"))
	void RunShaderMaterialAudit();

	/** Log a summary of common bottlenecks and actionable recommendations. */
	UFUNCTION(BlueprintCallable, Category = "RedShift PASYS", meta = (DisplayName = "Run Bottleneck Report"))
	void RunBottleneckReport();

	/** Set minimum texture size (MB) to report in texture audit. Default 1.0. */
	UPROPERTY(EditAnywhere, Category = "RedShift PASYS", meta = (ClampMin = "0.01", ClampMax = "100"))
	float TextureAuditMinSizeMB = 1.0f;

	/** Maximum number of texture entries to log in detail. 0 = unlimited. */
	UPROPERTY(EditAnywhere, Category = "RedShift PASYS", meta = (ClampMin = "0", ClampMax = "5000"))
	int32 TextureAuditMaxEntries = 100;

	/** Write report to Saved/RedShiftPASYS/ as well as Output Log. */
	UPROPERTY(EditAnywhere, Category = "RedShift PASYS")
	bool bWriteReportToFile = true;

private:
	void LogToReport(const FString& Line);
	void FlushReportToFile();
	FString ReportBuffer;
	FString CurrentReportFilename;
};
