// Copyright RedShift PASYS. All Rights Reserved.

#include "RedShiftPASYSSubsystem.h"
#include "Engine/Texture2D.h"
#include "Engine/TextureCube.h"
#include "Engine/TextureRenderTarget2D.h"
#include "Materials/Material.h"
#include "Materials/MaterialInstance.h"
#include "Materials/MaterialInterface.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Misc/Paths.h"
#include "Misc/FileHelper.h"
#include "HAL/PlatformFileManager.h"
#include "UObject/SavePackage.h"
#include "Engine/World.h"
#include "EngineUtils.h"
#include "Components/PrimitiveComponent.h"
#include "Engine/StaticMesh.h"
#include "RenderingThread.h"
#include "TextureResource.h"
#include "ProfilingDebugging/ResourceSize.h"

DEFINE_LOG_CATEGORY_STATIC(LogRedShiftPASYS, Log, All);

void URedShiftPASYSSubsystem::LogToReport(const FString& Line)
{
	UE_LOG(LogRedShiftPASYS, Log, TEXT("%s"), *Line);
	ReportBuffer += Line + TEXT("\n");
}

void URedShiftPASYSSubsystem::FlushReportToFile()
{
	if (!bWriteReportToFile || ReportBuffer.IsEmpty() || CurrentReportFilename.IsEmpty()) return;

	FString Dir = FPaths::ProjectSavedDir() / TEXT("RedShiftPASYS");
	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
	if (!PlatformFile.DirectoryExists(*Dir))
	{
		PlatformFile.CreateDirectoryTree(*Dir);
	}
	FString Path = Dir / CurrentReportFilename;
	FFileHelper::SaveStringToFile(ReportBuffer, *Path, FFileHelper::EEncodingOptions::ForceUTF8WithoutBOM);
	UE_LOG(LogRedShiftPASYS, Log, TEXT("Report saved to: %s"), *Path);
}

void URedShiftPASYSSubsystem::RunFullOptimizationReport()
{
	ReportBuffer.Empty();
	CurrentReportFilename = FString::Printf(TEXT("PASYS_Report_%s.txt"), *FDateTime::Now().ToString(TEXT("%Y%m%d_%H%M%S")));

	LogToReport(TEXT("========== RedShift PASYS - Full Optimization Report =========="));
	LogToReport(FDateTime::Now().ToString());
	LogToReport(TEXT(""));

	RunTextureAudit();
	LogToReport(TEXT(""));
	RunShaderMaterialAudit();
	LogToReport(TEXT(""));
	RunBottleneckReport();

	LogToReport(TEXT(""));
	LogToReport(TEXT("========== End Report =========="));
	FlushReportToFile();
}

void URedShiftPASYSSubsystem::RunTextureAudit()
{
	LogToReport(TEXT("--- Texture Audit ---"));

	FAssetRegistryModule& AssetRegistryModule = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	IAssetRegistry& AssetRegistry = AssetRegistryModule.Get();

	FARFilter Filter;
	Filter.ClassPaths.Add(FTopLevelAssetPath(TEXT("/Script/Engine"), TEXT("Texture2D")));
	Filter.bRecursiveClasses = true;
	TArray<FAssetData> TextureAssets;
	AssetRegistry.GetAssets(Filter, TextureAssets);

	const float MinSizeBytes = TextureAuditMinSizeMB * 1024.f * 1024.f;
	struct FTextureEntry
	{
		FString Path;
		int64 SizeBytes;
		int32 Width, Height;
		FString Format;
		bool bStreaming;
	};
	TArray<FTextureEntry> Entries;
	int64 TotalSize = 0;
	int32 NonStreamingCount = 0;
	int32 OverSizeCount = 0;

	for (const FAssetData& Data : TextureAssets)
	{
		UObject* Obj = Data.GetAsset();
		UTexture2D* Tex = Cast<UTexture2D>(Obj);
		if (!Tex || Tex->GetOuter() == nullptr) continue;

		FResourceSizeEx ResSize(EResourceSizeMode::EstimatedTotal);
		Tex->GetResourceSizeEx(ResSize);
		int64 SizeBytes = ResSize.GetDedicatedSystemMemoryBytes() + ResSize.GetDedicatedVideoMemoryBytes();
		if (SizeBytes <= 0)
		{
			ResSize = FResourceSizeEx(EResourceSizeMode::Exclusive);
			Tex->GetResourceSizeEx(ResSize);
			SizeBytes = ResSize.GetDedicatedSystemMemoryBytes() + ResSize.GetDedicatedVideoMemoryBytes();
		}
		TotalSize += SizeBytes;
		bool bStreaming = Tex->GetStreamingIndex() != INDEX_NONE;
		if (!bStreaming && Tex->LODGroup != TEXTUREGROUP_UI) NonStreamingCount++;
		if (SizeBytes >= MinSizeBytes)
		{
			OverSizeCount++;
			FTextureEntry E;
			E.Path = Data.GetSoftObjectPath().ToString();
			E.SizeBytes = SizeBytes;
			E.Width = Tex->GetSizeX();
			E.Height = Tex->GetSizeY();
			E.Format = FString::Printf(TEXT("%d"), (int32)Tex->GetPixelFormat());
			E.bStreaming = bStreaming;
			Entries.Add(E);
		}
	}

	// Sort by size descending
	Entries.Sort([](const FTextureEntry& A, const FTextureEntry& B) { return A.SizeBytes > B.SizeBytes; });

	LogToReport(FString::Printf(TEXT("Total textures scanned: %d"), TextureAssets.Num()));
	LogToReport(FString::Printf(TEXT("Total texture memory (estimated): %.2f MB"), TotalSize / (1024.0 * 1024.0)));
	LogToReport(FString::Printf(TEXT("Textures >= %.2f MB: %d"), TextureAuditMinSizeMB, OverSizeCount));
	LogToReport(FString::Printf(TEXT("Non-streaming (non-UI) textures: %d (consider enabling streaming for large textures)"), NonStreamingCount));
	LogToReport(TEXT(""));

	LogToReport(TEXT("Largest textures (actionable):"));
	int32 NumToShow = (TextureAuditMaxEntries > 0) ? FMath::Min(TextureAuditMaxEntries, Entries.Num()) : Entries.Num();
	for (int32 i = 0; i < NumToShow; i++)
	{
		const FTextureEntry& E = Entries[i];
		FString Rec = E.bStreaming ? TEXT("") : TEXT(" [REC: enable streaming or reduce size]");
		LogToReport(FString::Printf(TEXT("  %.2f MB | %dx%d | %s | %s%s"),
			E.SizeBytes / (1024.0 * 1024.0), E.Width, E.Height, *E.Format, *E.Path, *Rec));
	}

	LogToReport(TEXT(""));
	LogToReport(TEXT("Texture recommendations:"));
	LogToReport(TEXT("  - Use texture streaming for large world/level textures."));
	LogToReport(TEXT("  - Prefer BC/DXT compression for diffuse/normal where possible."));
	LogToReport(TEXT("  - Reduce resolution for distant or small-on-screen assets."));
	LogToReport(TEXT("  - Use texture LOD and mip maps; avoid NoMipmaps unless needed."));
}

void URedShiftPASYSSubsystem::RunShaderMaterialAudit()
{
	LogToReport(TEXT("--- Shader / Material Audit ---"));

	FAssetRegistryModule& AssetRegistryModule = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	IAssetRegistry& AssetRegistry = AssetRegistryModule.Get();

	FARFilter Filter;
	Filter.ClassPaths.Add(FTopLevelAssetPath(TEXT("/Script/Engine"), TEXT("Material")));
	Filter.bRecursiveClasses = true;
	TArray<FAssetData> MaterialAssets;
	AssetRegistry.GetAssets(Filter, MaterialAssets);

	int32 TotalMaterials = 0;
	int32 ComplexCount = 0;
	TArray<TPair<FString, int32>> ComplexEntries;

	for (const FAssetData& Data : MaterialAssets)
	{
		UMaterial* Mat = Cast<UMaterial>(Data.GetAsset());
		if (!Mat) continue;
		TotalMaterials++;
		bool bTwoSided = Mat->IsTwoSided();
		bool bTranslucent = (Mat->GetBlendMode() == BLEND_Translucent || Mat->GetBlendMode() == BLEND_AlphaComposite);
		if (bTwoSided || bTranslucent)
		{
			ComplexCount++;
			ComplexEntries.Add(TPair<FString, int32>(Data.GetSoftObjectPath().ToString(), (bTwoSided ? 1 : 0) + (bTranslucent ? 2 : 0)));
		}
	}

	LogToReport(FString::Printf(TEXT("Materials scanned: %d"), TotalMaterials));
	LogToReport(FString::Printf(TEXT("Potentially complex / expensive: %d"), ComplexCount));
	LogToReport(TEXT(""));
	LogToReport(TEXT("Shader/Material recommendations:"));
	LogToReport(TEXT("  - Prefer opaque materials; use masked/translucent only when needed."));
	LogToReport(TEXT("  - Reduce material instruction count; merge similar materials."));
	LogToReport(TEXT("  - Use material instances instead of full materials where possible."));
	LogToReport(TEXT("  - Disable two-sided where not needed to save overdraw."));
	LogToReport(TEXT("  - Use shader complexity view (Show > Shader Complexity) to find hotspots."));
}

void URedShiftPASYSSubsystem::RunBottleneckReport()
{
	LogToReport(TEXT("--- Bottleneck & Solvable Problems Summary ---"));

	LogToReport(TEXT("Common bottlenecks and fixes:"));
	LogToReport(TEXT("  1) GPU-bound: Reduce draw calls (merge static meshes, LODs), simplify shaders, lower shadow/reflection quality."));
	LogToReport(TEXT("  2) CPU-bound: Reduce tick cost, use fewer dynamic lights, simplify physics/collision."));
	LogToReport(TEXT("  3) Memory: Downsize textures, enable streaming, reduce material/texture duplication."));
	LogToReport(TEXT("  4) Streaming: Ensure world textures are in streaming pools; avoid loading entire levels at once."));
	LogToReport(TEXT("  5) Shader compilation hitches: Use 'Precompile Shaders' or cook with -SkipShaderCompilation=false for target."));
	LogToReport(TEXT(""));
	LogToReport(TEXT("Use Unreal Insights / stat GPU / stat Game for live profiling."));
	LogToReport(TEXT("RedShift PASYS reports are saved under Saved/RedShiftPASYS/ when bWriteReportToFile is true."));
}
