// Copyright RedShift PASYS. All Rights Reserved.

#include "RedShiftPASYSModule.h"
#include "RedShiftPASYSSubsystem.h"
#include "RedShiftPASYSStyle.h"
#include "Framework/MultiBox/MultiBoxBuilder.h"
#include "LevelEditor.h"
#include "Editor.h"
#include "Misc/MessageDialog.h"
#include "Styling/AppStyle.h"

#define LOCTEXT_NAMESPACE "RedShiftPASYS"

void FRedShiftPASYSModule::StartupModule()
{
	FRedShiftPASYSStyle::Initialize();
	FRedShiftPASYSStyle::ReloadTextures();
	RegisterMenus();

	FLevelEditorModule& LevelEditorModule = FModuleManager::LoadModuleChecked<FLevelEditorModule>("LevelEditor");
	LevelEditorModule.GetMenuExtensibilityManager()->AddExtender(MenuExtender.ToSharedRef());
	LevelEditorModule.GetToolBarExtensibilityManager()->AddExtender(ToolBarExtender.ToSharedRef());
}

void FRedShiftPASYSModule::ShutdownModule()
{
	UnregisterMenus();
	FRedShiftPASYSStyle::Shutdown();
}

void FRedShiftPASYSModule::RegisterMenus()
{
	PluginCommands = MakeShareable(new FUICommandList);

	MenuExtender = MakeShareable(new FExtender);
	MenuExtender->AddMenuExtension(
		"WindowLayout",
		EExtensionHook::After,
		PluginCommands,
		FMenuExtensionDelegate::CreateRaw(this, &FRedShiftPASYSModule::AddMenuEntry));

	ToolBarExtender = MakeShareable(new FExtender);
	ToolBarExtender->AddToolBarExtension(
		"Settings",
		EExtensionHook::After,
		PluginCommands,
		FToolBarExtensionDelegate::CreateRaw(this, &FRedShiftPASYSModule::AddToolBarEntry));
}

void FRedShiftPASYSModule::UnregisterMenus()
{
}

void FRedShiftPASYSModule::AddMenuEntry(FMenuBuilder& MenuBuilder)
{
	MenuBuilder.AddSubMenu(
		LOCTEXT("RedShiftPASYSMenu", "RedShift PASYS"),
		LOCTEXT("RedShiftPASYSTooltip", "Performance Analysis: textures, shaders, bottlenecks"),
		FNewMenuDelegate::CreateLambda([this](FMenuBuilder& SubMenu)
		{
			SubMenu.AddMenuEntry(
				LOCTEXT("FullReport", "Run Full Optimization Report"),
				LOCTEXT("FullReportTip", "Run texture, shader, and bottleneck report and save to file"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon"),
				FUIAction(FExecuteAction::CreateLambda([this]()
				{
					URedShiftPASYSSubsystem* Sub = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>();
					if (Sub) { Sub->RunFullOptimizationReport();
						FMessageDialog::Open(EAppMsgType::Ok, LOCTEXT("ReportDone", "RedShift PASYS: Full report done. Check Output Log and Saved/RedShiftPASYS/.")); }
				})));
			SubMenu.AddMenuEntry(
				LOCTEXT("TextureAudit", "Run Texture Audit"),
				LOCTEXT("TextureAuditTip", "List large textures and streaming recommendations"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon16"),
				FUIAction(FExecuteAction::CreateLambda([]() { if (URedShiftPASYSSubsystem* S = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>()) S->RunTextureAudit(); })));
			SubMenu.AddMenuEntry(
				LOCTEXT("BottleneckReport", "Run Bottleneck Report"),
				LOCTEXT("BottleneckReportTip", "Log common bottlenecks and solvable fixes"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon16"),
				FUIAction(FExecuteAction::CreateLambda([]() { if (URedShiftPASYSSubsystem* S = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>()) S->RunBottleneckReport(); })));
		}),
		false,
		FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon"));
}

void FRedShiftPASYSModule::AddToolBarEntry(FToolBarBuilder& ToolBarBuilder)
{
	ToolBarBuilder.AddComboButton(
		FUIAction(),
		FOnGetContent::CreateLambda([this]()
		{
			FMenuBuilder MenuBuilder(true, PluginCommands);
			MenuBuilder.AddMenuEntry(
				LOCTEXT("FullReport", "Run Full Optimization Report"),
				LOCTEXT("FullReportTip", "Run full report"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon"),
				FUIAction(FExecuteAction::CreateLambda([]()
				{
					if (URedShiftPASYSSubsystem* Sub = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>())
					{
						Sub->RunFullOptimizationReport();
						FMessageDialog::Open(EAppMsgType::Ok, LOCTEXT("ReportDone2", "RedShift PASYS: Full report done. Check Output Log and Saved/RedShiftPASYS/."));
					}
				})));
			MenuBuilder.AddMenuEntry(
				LOCTEXT("TextureAudit", "Run Texture Audit"),
				LOCTEXT("TextureAuditTip", "Texture audit"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon16"),
				FUIAction(FExecuteAction::CreateLambda([]() { if (URedShiftPASYSSubsystem* S = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>()) S->RunTextureAudit(); })));
			MenuBuilder.AddMenuEntry(
				LOCTEXT("BottleneckReport", "Run Bottleneck Report"),
				LOCTEXT("BottleneckReportTip", "Bottleneck report"),
				FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon16"),
				FUIAction(FExecuteAction::CreateLambda([]() { if (URedShiftPASYSSubsystem* S = GEditor->GetEditorSubsystem<URedShiftPASYSSubsystem>()) S->RunBottleneckReport(); })));
			return MenuBuilder.MakeWidget();
		}),
		LOCTEXT("RedShiftPASYS_Label", "RedShift PASYS"),
		LOCTEXT("RedShiftPASYS_Tooltip", "Performance Analysis System"),
		FSlateIcon(FRedShiftPASYSStyle::GetStyleSetName(), "RedShiftPASYS.Icon"));
}

#undef LOCTEXT_NAMESPACE

IMPLEMENT_MODULE(FRedShiftPASYSModule, RedShiftPASYS)
